/******************************************************************************
* NwkAppInit.c
*
* Initialization common to all applications. The very start of the program,
* main(), is found here.
*
* Copyright (c) 2008, Freescale, Inc. All rights reserved.
*
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor
.
*******************************************************************************/
#include "EmbeddedTypes.h"
#include "MsgSystem.h"
#include "Uart_Interface.h"
#include "AppAspInterface.h"
#include "NwkMacInterface.h"
#include "NVM_Interface.h"
#include "NV_Flash.h"
#include "NwkInit.h"
#include "NwkApp.h"
#include "NwkAppInit.h"
#include "PublicConst.h"
#include "Crt0.h"
#include "TS_Interface.h"
#include "IrqControlLib.h"
#include "PlatformInit.h"
#include "TMR_Interface.h"
#include "pwr_interface.h"
#include "ApplicationConf.h"
#include "ApplicationFunctionalityDefines.h"
#include "UartUtil.h"
#include "FSLProfileGlobals.h"
#include "ZRCProfileGlobals.h"
#include "FSLProfileInterface.h"
#include "ZRCProfileInterface.h"
#if gFSLProfileOtapClient_d
  #include "OtapSupport.h"
#endif


/******************************************************************************
*******************************************************************************
* Private macros
*******************************************************************************
******************************************************************************/


/******************************************************************************
*******************************************************************************
* Public prototypes
*******************************************************************************
******************************************************************************/


/******************************************************************************
*******************************************************************************
* Private prototypes
*******************************************************************************
******************************************************************************/
static void HandleLowPower(void);

/******************************************************************************
*******************************************************************************
* Public memory declarations
*******************************************************************************
******************************************************************************/
/* Identifier of the application task */
tsTaskID_t gAppTaskID;


/******************************************************************************
*******************************************************************************
* Public functions
*******************************************************************************
******************************************************************************/


/*****************************************************************************
* IdleTask
*
* Called by the kernel when no other task has an event to process.
*
*****************************************************************************/
void IdleTask(event_t events)
{

  (void)events; /* remove compiler warning */

  /* There are some UART errors that are hard to clear in the UART */
  /* ISRs, and the UART driver does not have a task to clear them */
  /* in non-interrupt context. */
  Uart_ClearErrors();

  if((NWK_IsIdle() == TRUE)
#if gZRCProfileCommandTxRx_d  
     && (ZRCProfile_IsIdle())
#endif  
#if gPBPTask_d  
     && (PBP_IsIdle())
#endif   
#if gFSLProfileTask_d
     && (FSLProfile_IsIdle())
#endif  /* gFSLProfileTask_d */

     )
  {
#if gNvStorageIncluded_d
	      /* Process NV Storage save-on-idle and save-on-count requests. */
	      NvIdle();
	      /* There is a new image availlable,write flags in flash */
  #if gFSLProfileOtapClient_d	      
	      OTAP_WriteNewImageFlashFlags();
	#endif      
#endif /* gNvStorageIncluded_d */

#if gLpmIncluded_d
        HandleLowPower();
#endif /* gLpmIncluded_d */

  }
}                                     

/******************************************************************************
*******************************************************************************
* Private functions
*******************************************************************************
******************************************************************************/

void main(void) {
  /* Disable all interrupts */
  IrqControlLib_DisableAllIrqs();  
  /* Init platform */
  Platform_Init();   
  /* Initialize 802.15.4 MAC and PHY */
  Init_802_15_4();
  /* Initialize the operation system kernel */
  TS_Init();                            
  /* Initialize the platform timer module */
  TMR_Init();
  /* Initialize RF4CE layer */
  NwkInit();
  /* Create the application task */  
  gAppTaskID          = TS_CreateTask(gTsAppTaskPriority_c, App_MainTask);
  
#if gFSLProfileTask_d
  /* Create the FSL Profile task */ 
  gFSLProfileTaskID_c = TS_CreateTask(gTsFSLProfileTaskPriority_c, FSLProfile_MainTask);
  /* Initialize the FSL Profile */
  FSLProfile_Init();
#endif /* gFSLProfileTask_d */
  
#if gZRCProfileCommandTxRx_d
  /* Create the ZRC Profile task */ 
  gZRCProfileTaskID_c = TS_CreateTask(gTsZRCProfileTaskPriority_c, ZRCProfile_MainTask);
  /* Initialize the ZRC Profile */
  ZRCProfile_Init();
#endif /* gZRCProfileCommandTxRx_d */
  
#if gPBPTask_d   
  /* Create the PBP task */ 
  gPBPTaskID_c = TS_CreateTask(gTsPushButtonTaskPriority_c, PBP_MainTask);
  /* Initialize the PBP */
  PBP_Init();
#endif  
  
  /* Initialize the application over RF4CE layer */
  App_Init();
  
  /* Start the task scheduler. Does not return. */
  TS_Scheduler();
}

/******************************************************************************
*******************************************************************************
* Debugging and Unit Tests
*******************************************************************************
******************************************************************************/

/* Called each time deep sleep mode is exited. */
void DeepSleepWakeupStackProc(void)
{		
  return;
}

/******************************************************************************
* BeeAppAssert
*
* Something failed in the stack or app that can't recover. Stop here. Mostly
* used during debug phase of product. Could be used
******************************************************************************/
void BeeAppAssert(assertCode_t code)
{
  volatile bool_t waitForever = code; /* just set to anything non-zero */
  
  while (waitForever)
  { 
  }
}

/******************************************************************************
* HandleLowPower
*
* This function tries to put the MCU and the transceiver in low power modes
* if this is requested by the application
******************************************************************************/
static void HandleLowPower(void)
{
  PWRLib_WakeupReason_t WakeupReason;
  /* Enter low power mode if requested by application */          
  if(PWR_CheckIfDeviceCanGoToSleep())
  {        
    uint32_t timeToStayInLowPower = NWK_GetAllowedLowPowerInterval();     
    
    /* Try to enter low power mode only if the nwk allows it */
    if(timeToStayInLowPower)
    {
      /* Set the amount of time to stay in low power mode */
      PWR_SetDozeTimeSymbols(timeToStayInLowPower);
      /* Clear the bit field cotaining the wake up reason */
      PWRLib_MCU_WakeupReason.AllBits = 0;
      /* Try to enter low power mode. This is a blocking function call */
      
      WakeupReason = PWR_EnterLowPower();
      
      /* Test if device has been waken up from low power by a keyboard press */    
      if(WakeupReason.Bits.FromKBI)    
      {
        /* At this point, the MCU and radio have exited the low power mode. */
        UartUtil_Print("\n\rDevice woken by keyboard interrupt.\n\r",gAllowToBlock_d);
        /* Do not allow device to enter low power mode again */     
        PWR_DisallowDeviceToSleep();
      }
      /*@AN4398 BEGIN: Comparison added to handle RTI and SCI wake up sources*/
      if(WakeupReason.Bits.FromRTI)    
		{
		  /* At this point, the MCU and radio have exited the low power mode. */
		  UartUtil_Print("\n\rDevice woken by RTI interrupt.\n\r",gAllowToBlock_d);
		  /* Do not allow device to enter low power mode again */     
		  PWR_DisallowDeviceToSleep();
		}
      if(WakeupReason.Bits.FromSCI)    
		{
		  /* At this point, the MCU and radio have exited the low power mode. */
		  UartUtil_Print("\n\rDevice woken by SCI interrupt.\n\r",gAllowToBlock_d);
		  /* Do not allow device to enter low power mode again */     
		  PWR_DisallowDeviceToSleep();
		}
      /*@AN4398 END: Comparison added to handle RTI and SCI wake up sources*/
      
    }
  }
}
